// File: LEDblinker.c
// This file contains all LED blinker implementation
// Author: dr. Matej Kseneman

// Includes
#include "STM32F4_Discovery.h"
#include "LEDblinker.h"

// Defines
#define FALSE		0
#define TRUE		1

// Global variables
GPIO_InitTypeDef GPIO_InitStructure;
NVIC_InitTypeDef NVIC_InitStructure;
EXTI_InitTypeDef EXTI_InitStructure;

const uint16_t ExternLeds[] = { RED_LED, ORANGE_LED, GREEN_LED, BLUE_LED };
const uint8_t UserLeds[] = { LED5, LED3, LED4, LED6 };
volatile int idx = -1;

// Function pointer
uint8_t isClockwise = TRUE;
void (*toggleLed)(void);

// Define user button interrupt
// User button is on PA0
void ConfigUserButton()
{
	// Enable GPIOA and SYSCFG clock
	RCC_AHB1PeriphClockCmd(USER_BUTTON_GPIO_CLK, ENABLE);

	// Configure PA0 pin as input floating
	GPIO_InitStructure.GPIO_Pin = USER_BUTTON_PIN;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_IN;
	GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
	GPIO_Init(USER_BUTTON_GPIO_PORT, &GPIO_InitStructure);

	// Connect EXTI Line0 to PA0 pin
	SYSCFG_EXTILineConfig(USER_BUTTON_EXTI_PORT_SOURCE, USER_BUTTON_EXTI_PIN_SOURCE);

	// Configure EXTI Line0
	EXTI_InitStructure.EXTI_Line = USER_BUTTON_EXTI_LINE;
	EXTI_InitStructure.EXTI_Mode = EXTI_Mode_Interrupt;
	EXTI_InitStructure.EXTI_Trigger = EXTI_Trigger_Rising;
	EXTI_InitStructure.EXTI_LineCmd = ENABLE;
	EXTI_Init(&EXTI_InitStructure);

	// Enable and set EXTI Line0 Interrupt to the lowest priority
	NVIC_InitStructure.NVIC_IRQChannel = USER_BUTTON_EXTI_IRQn;
	NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0x01;
	NVIC_InitStructure.NVIC_IRQChannelSubPriority = 0x01;
	NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
	NVIC_Init(&NVIC_InitStructure);
}

// Initialize periphery
void InitBlinker()
{
	uint8_t i;

	// First initialize on board LEDs
	for (i = 0; i < sizeof(UserLeds); ++i)
	{
		STM_EVAL_LEDInit(i);
	}

	// Enable port clock
	RCC_AHB1PeriphClockCmd(LEDS_GPIO_CLK, ENABLE);

	// Create pin definition for external LEDs
	GPIO_InitStructure.GPIO_Pin = ALL_EXTERNAL_LEDS;
	GPIO_InitStructure.GPIO_Mode = GPIO_Mode_OUT;
	GPIO_InitStructure.GPIO_OType = GPIO_OType_PP;
	GPIO_InitStructure.GPIO_Speed = GPIO_Speed_2MHz;
	GPIO_InitStructure.GPIO_PuPd = GPIO_PuPd_NOPULL;
	GPIO_Init(LEDS_GPIO_PORT, &GPIO_InitStructure);

	// Set clockwise direction first
	toggleLed = &ToggleLedForward;

	// Configure User button interrupt
	ConfigUserButton();
}

// Toggle LED state in a clockwise direction
void ToggleLedForward()
{
	++idx;
	idx %= sizeof(UserLeds);

	// Toggle LEDs
	STM_EVAL_LEDToggle(UserLeds[idx]);
	GPIO_ToggleBits(LEDS_GPIO_PORT, ExternLeds[idx]);
}

// Toggle LED state in a counterclockwise direction
void ToggleLedBackward()
{
	if (idx < 0)
	{
		idx += sizeof(UserLeds);
	}

	// Toggle LEDs
	STM_EVAL_LEDToggle(UserLeds[idx]);
	GPIO_ToggleBits(LEDS_GPIO_PORT, ExternLeds[idx]);
	--idx;
}

// Change LEDs direction
// This function is fired upon User button interrupt
void ChangeLedDirection()
{
	if (isClockwise)
	{
		toggleLed = &ToggleLedBackward;
		isClockwise = FALSE;
	}
	else
	{
		toggleLed = &ToggleLedForward;
		isClockwise = TRUE;
	}
}
