// File: delay.c
// This file contains delay implementation
// Author: dr. Matej Kseneman

// Includes
#include "stm32f4xx_tim.h"
#include "delay.h"

// Defines
#define MILLI_SECOND    1000
#define MICRO_SECOND    1000000
#define SHORT_VERSION	0

// Global variables
volatile uint32_t TimerOverflow = 0;

/**
 * @brief  Initialize timer
 * @param  None
 * @retval None
 */
void InitUsTimer()
{
	// Initialize 1 microsecond timer
	RCC_ClocksTypeDef RCC_Clocks;
	NVIC_InitTypeDef NVIC_InitStructure;
	RCC_GetClocksFreq(&RCC_Clocks);

#if SHORT_VERSION
	// Write directly to registers
	RCC->APB1ENR |= RCC_APB1Periph_TIM7;
	TIM7->PSC = (RCC_Clocks.PCLK1_Frequency / 500000) - 1; // 2 * APB1 (Hz) / 1 MHz
	TIM7->ARR = 0xFFFF;
	TIM7->CR1 = TIM_CR1_CEN;
#else
	TIM_TimeBaseInitTypeDef TIM_TimeBaseStructure;

	RCC_APB1PeriphClockCmd(RCC_APB1Periph_TIM7, ENABLE);

	// Time base configuration */
	TIM_TimeBaseStructure.TIM_Period = 0xFFFF;
	TIM_TimeBaseStructure.TIM_Prescaler = (RCC_Clocks.PCLK1_Frequency / 500000) - 1;
	TIM_TimeBaseStructure.TIM_ClockDivision = 0;
	TIM_TimeBaseStructure.TIM_CounterMode = TIM_CounterMode_Up;
	TIM_TimeBaseInit(TIM7, &TIM_TimeBaseStructure);

	// TIM 7 enable counter
	TIM_Cmd(TIM7, ENABLE);
#endif

	// Enable the TIM7 global Interrupt
	NVIC_InitStructure.NVIC_IRQChannel = TIM7_IRQn;
	NVIC_InitStructure.NVIC_IRQChannelPreemptionPriority = 0;
	NVIC_InitStructure.NVIC_IRQChannelSubPriority = 1;
	NVIC_InitStructure.NVIC_IRQChannelCmd = ENABLE;
	NVIC_Init(&NVIC_InitStructure);

	// TIM IT enable
	TIM_ITConfig(TIM7, TIM_IT_Update, ENABLE);
}

// Get time in microsecond ticks
/* Because this timer is 16-bit, we need to call this
 * function every 65 millisecond (0x10000 us).
 */
uint64_t GetMicroSecondTime()
{
	// Return current tick count
	return ((uint64_t) TimerOverflow << 16) + TIM7->CNT;
}

/**
 * @brief  Inserts a delay time.
 * @param  nTime: specifies the delay time length, in milliseconds.
 * @retval None
 */
void DelayMilliSeconds(uint32_t nTime)
{
	DelayMicroSeconds((uint64_t)nTime * 1000);
}

/**
 * Perform a microsecond delay
 *
 * \param  usTime	Number of microseconds to wait.
 */
void DelayMicroSeconds(uint64_t usTime)
{
	// Calculate end time
	uint64_t endTime = GetMicroSecondTime() + usTime;

	while (GetMicroSecondTime() < endTime);
}
